<?php

/**
 * WooCommerce MailChimp Pro plugin main class
 */
final class SS_WC_MailChimp_Pro_Plugin {

	/**
	 * Plugin version
	 *
	 * @var string
	 */
	private static $version = '2.4.4';

	/**
	 * Plugin singleton instance
	 *
	 * @var SS_WC_MailChimp_Pro_Plugin
	 */
	private static $instance;

	/**
	 * Plugin namespace
	 * @var string
	 */
	private $namespace = 'ss_wc_mailchimp_pro';

	/**
	 * Plugin settings
	 * @var array
	 */
	private $settings;

	/**
	 * Plugin license handler
	 * @var SS_WC_MailChimp_Pro_License_Handler
	 */
	private $license_handler;

	/**
	 * Plugin MailChimp helper instance
	 * @var SS_WC_MailChimp
	 */
	private $mailchimp;

	/**
	 * Plugin compatibility checker
	 * @return SS_WC_MailChimp_Compatibility
	 */
	public $compatibility;

	/**
	 * Returns the plugin version
	 * @return string
	 */
	public static function version() {
		return self::$version;
	}

	/**
	 * Singleton instance
	 *
	 * @return SS_WC_MailChimp_Pro_Plugin   SS_WC_MailChimp_Pro_Plugin object
	 */
	public static function get_instance() {

		if ( empty( self::$instance ) && ! ( self::$instance instanceof SS_WC_MailChimp_Pro_Plugin ) ) {

			self::$instance = new SS_WC_MailChimp_Pro_Plugin;
			self::$instance->define_constants();

			self::$instance->includes();
			self::$instance->compatibility = SS_WC_MailChimp_Pro_Compatibility::get_instance();
			self::$instance->admin_notices = new SS_WC_MailChimp_Pro_Admin_Notices;

			if ( self::$instance->compatibility->is_valid() ) {
				self::$instance->handler = SS_WC_MailChimp_Pro_Handler::get_instance();
				self::$instance->subscriptions = SS_WC_MailChimp_Pro_Subscriptions::get_instance();
				self::$instance->load_plugin_textdomain();

				self::update();
				self::$instance->add_hooks();
				SS_WC_MailChimp_Pro_Settings::get_instance();
				self::$instance->license_handler = SS_WC_MailChimp_Pro_License_Handler::get_instance();
				do_action( 'ss_wc_mailchimp_pro_loaded' );
			}

		}

		return self::$instance;

	} //end function instance

	public function get_license_handler() {
		return $this->license_handler;
	}

	/**
	 * Gets the plugin db settings
	 * @param  boolean $refresh refresh the settings from DB?
	 * @return array  The plugin db settings
	 */
	public function settings( $refresh = false ) {

		if ( empty( $this->settings ) || true === $refresh ) {

			$defaults = require( SS_WC_MAILCHIMP_PRO_DIR . 'config/default-settings.php' );
			$settings = array();

			foreach ( $defaults as $key => $default_value ) {

				$setting_value = get_option( $this->namespace_prefixed( $key ) );

				$settings[ $key ] = $setting_value ? $setting_value : $default_value;
			}

			$merged_settings = array_merge( $defaults, $settings );

			$this->settings = $merged_settings;

		}

		return $this->settings;
	}

	/**
	 * Whether or not to always subscribe to product-specific lists (regardless of global "main list" settings)
	 *
	 * @access public
	 * @return boolean
	 */
	public function always_subscribe_to_product_lists() {
		return 'yes' === $this->settings['always_subscribe_to_product_lists'];
	}

	/**
	 * Saves the settings back to the DB
	 * @return void
	 */
	public function save_settings( $settings = null ) {

		if ( empty( $settings ) ) {

			$settings = $this->settings();

		}

		foreach ( $settings as $key => $value ) {
			update_option( $this->namespace_prefixed( $key ), $value );
		}

	} //end function save_settings

	/**
	 * Define Plugin Constants.
	 */
	private function define_constants() {

		// Minimum supported version of WooCommerce MailChimp
		$this->define( 'SS_WC_MAILCHIMP_MIN_WCMC_VERSION', '2.3.8' );

		// Plugin version.
		$this->define( 'SS_WC_MAILCHIMP_PRO_VERSION', self::version() );

		// Plugin Folder Path.
		$this->define( 'SS_WC_MAILCHIMP_PRO_DIR', plugin_dir_path( SS_WC_MAILCHIMP_PRO_FILE ) );

		// Plugin Folder URL.
		$this->define( 'SS_WC_MAILCHIMP_PRO_URL', plugin_dir_url( SS_WC_MAILCHIMP_PRO_FILE ) );

		$settings_url = admin_url( 'admin.php?page=wc-settings&tab=mailchimp&section=pro' );

		// Plugin Settings URL
		$this->define( 'SS_WC_MAILCHIMP_PRO_SETTINGS_URL', $settings_url );

	} //function define_constants

	/**
	 * Define constant if not already set.
	 *
	 * @param  string $name
	 * @param  string|bool $value
	 */
	private function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	} //function define

	/**
	 * Include required core plugin files
	 */
	public function includes() {

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-compatibility.php' );

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-admin-notices.php' );

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-settings.php' );

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-license-handler.php');

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-handler.php' );

		require_once( SS_WC_MAILCHIMP_PRO_DIR . 'includes/class-ss-wc-mailchimp-pro-subscriptions.php' );

	} //end function includes

	/**
	 * Load Localization files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/plugins/woocommerce-mailchimp/woocommerce-mailchimp-{lang}_{country}.mo
	 *      - WP_CONTENT_DIR/plugins/woocommerce-mailchimp/languages/woocommerce-mailchimp-{lang}_{country}.mo
	 */
	public function load_plugin_textdomain() {

		// Set filter for plugin's languages directory.
		$woocommerce_mailchimp_lang_dir  = dirname( plugin_basename( SS_WC_MAILCHIMP_PRO_FILE ) ) . '/languages/';

		// Traditional WordPress plugin locale filter.
		// get locale in {lang}_{country} format (e.g. en_US)
		$locale = apply_filters( 'plugin_locale', get_locale(), 'woocommerce-mailchimp' );

		$mofile = sprintf( '%1$s-%2$s.mo', 'woocommerce-mailchimp', $locale );

		// Look for wp-content/languages/woocommerce-mailchimp/woocommerce-mailchimp-{lang}_{country}.mo
		$mofile_global1 = WP_LANG_DIR . '/woocommerce-mailchimp/' . $mofile;

		// Look in wp-content/languages/plugins/woocommerce-mailchimp
		$mofile_global2 = WP_LANG_DIR . '/plugins/woocommerce-mailchimp/' . $mofile;

		if ( file_exists( $mofile_global1 ) ) {

			load_textdomain( 'woocommerce-mailchimp', $mofile_global1 );

		} elseif ( file_exists( $mofile_global2 ) ) {

			load_textdomain( 'woocommerce-mailchimp', $mofile_global2 );

		} else {

			// Load the default language files.
			load_plugin_textdomain( 'woocommerce-mailchimp', false, $woocommerce_mailchimp_lang_dir );

		}

	} //end function load_plugin_textdomain

	/**
	 * Add plugin hooks
	 */
	private function add_hooks() {

		// Add the "Settings" links on the Plugins administration screen.
		if ( is_admin() ) {

			if ( defined( 'SS_WC_MAILCHIMP_SETTINGS_URL' ) ) {

				add_action( 'after_plugin_row_woocommerce-mailchimp-pro/woocommerce-mailchimp-pro.php', array( $this, 'plugin_row' ) );
				add_filter( 'plugin_action_links_' . plugin_basename( SS_WC_MAILCHIMP_PRO_FILE ), array( $this, 'action_links' ) );

			}

			add_filter( 'plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 2 );

			add_action( 'admin_enqueue_scripts', array( $this, 'enqueue_scripts' ) );

		}

		// add_action( 'woocommerce_edit_account_form', array( $this, 'add_manage_subscription_box' ), 10, 2 );

	} //end function add_hooks

	public function add_manage_subscription_box() {
		$sswcmc = SSWCMC();
		?>
		<p class="woocommerce-form-row woocommerce-form-row--wide form-row form-row-wide">
			<label for="ss_wc_mailchimp_opt_in"><input type="checkbox" class="woocommerce-Input woocommerce-Input--checkbox input-checkbox" name="ss_wc_mailchimp_opt_in" id="ss_wc_mailchimp_opt_in" value="yes" <?php echo $sswcmc->opt_in_checkbox_default_status() == 'checked' ? ' checked="checked"' : '' ?>/>&nbsp;<?php echo( $sswcmc->opt_in_label() ); ?></label>
		</p>
		<?php
	}

	/**
	 * Add Settings link to plugins list
	 *
	 * @param  array $links Plugin links
	 * @return array        Modified plugin links
	 */
	public function action_links( $links ) {
		$plugin_links = array(
			'<a href="' . SS_WC_MAILCHIMP_PRO_SETTINGS_URL . '">' . __( 'Settings', 'woocommerce-mailchimp' ) . '</a>',
		);

		return array_merge( $plugin_links, $links );

	} //end function action_links

	/**
	 * Show row meta on the plugin screen.
	 *
	 * @param mixed $links Plugin Row Meta.
	 * @param mixed $file  Plugin Base file.
	 *
	 * @return array
	 */
	public static function plugin_row_meta( $links, $file ) {
		if ( plugin_basename( SS_WC_MAILCHIMP_PRO_FILE ) === $file ) {
			$row_meta = array(
				'docs' => '<a href="' . esc_url( apply_filters( 'ss_wc_mailchimp_pro_documentation_url', 'https://support.saintsystems.com/hc/en-us/sections/201959566' ) ) . '" aria-label="' . esc_attr__( 'Documentation', 'woocommerce-mailchimp' ) . '" target="_blank">' . esc_html__( 'Documentation', 'woocommerce-mailchimp' ) . '</a>',
				'support' => '<a href="' . esc_url( apply_filters( 'ss_wc_mailchimp_pro_support_url', 'https://support.saintsystems.com/hc/en-us/requests/new' ) ) . '" aria-label="' . esc_attr__( 'Get Support', 'woocommerce-mailchimp' ) . '" target="_blank">' . esc_html__( 'Support', 'woocommerce-mailchimp' ) . '</a>',
			);

			return array_merge( $links, $row_meta );
		}

		return (array) $links;
	}

	/**
	 * Displays message on plugins page
	 * @param  string $plugin_name name of the plugin
	 * @return void
	 */
	public function plugin_row( $plugin_name ) {

		$settings = $this->settings( true );

		if ( ! $settings['license_key'] || $settings['license_key_status'] != 'valid' ) {

			echo '</tr><tr class="plugin-update-tr"><td colspan="3" class="plugin-update"><div class="update-message">' . __( sprintf( '%sRegister%s your copy of WooCommerce MailChimp Pro to receive access to automatic upgrades and support. Need a license key? %sPurchase one now%s.', '<a href="' . SS_WC_MAILCHIMP_PRO_SETTINGS_URL . '">', '</a>', '<a href="http://www.saintsystems.com/products/woocommerce-mailchimp-pro/#utm_source=woocommerce-mailchimp-pro-plugin-settings-link&utm_medium=textlink&utm_campaign=purchase-link" target="_blank">', '</a>' ), 'woocommerce-mailchimp' ) . '</div></td>';
		}

	} //end function plugin_row

	/**
     * Load scripts required for admin
     *
     * @access public
     * @return void
     */
    public function enqueue_scripts() {

		$screen       = get_current_screen();
		$screen_id    = $screen ? $screen->id : '';

    	// Plugin scripts and styles
		wp_register_script( 'woocommerce-mailchimp-pro-admin', SS_WC_MAILCHIMP_PRO_URL . 'assets/js/woocommerce-mailchimp-pro-admin.js', array( 'woocommerce-mailchimp-admin' ), self::version() );
		wp_register_style( 'woocommerce-mailchimp-pro', SS_WC_MAILCHIMP_PRO_URL . 'assets/css/style.css', array( 'woocommerce-mailchimp' ), self::version() );

		// Localize javascript messages
		// $translation_array = array(
		// 	'connecting_to_mailchimp'       => __( 'Connecting to MailChimp', 'woocommerce-mailchimp' ),
		// 	'error_loading_account'         => __( 'Error. Please check your api key.', 'woocommerce-mailchimp' ),
		// 	'error_loading_groups'          => __( 'Error loading groups. Please check your MailChimp Interest Groups for the selected list.', 'woocommerce-mailchimp' ),
		// 	'select_groups_placeholder'     => __( 'Select one or more groups (optional)', 'woocommerce-mailchimp' ),
		// 	'interest_groups_not_enabled'   => __( 'This list does not have interest groups enabled', 'woocommerce-mailchimp' ),
		// );
		//wp_localize_script( 'woocommerce-mailchimp-admin', 'SS_WC_MailChimp_Messages', $translation_array );

		// Scripts
		wp_enqueue_script( 'woocommerce-mailchimp-pro-admin' );

		if ( in_array( $screen_id, array( 'product', 'edit-product' ) ) ) {
			wc_enqueue_js("
				(function($){

					$(document).ready(function() {
						SS_WC_MailChimp_Pro.initProductTab();
					});

				})(jQuery);
			");
		}

		// Styles
		wp_enqueue_style( 'woocommerce-mailchimp-pro' );

		$this->license_handler->localize_scripts();

	} //end function enqueue_scripts

	/**
	 * Handles running plugin upgrades if necessary
	 *
	 * @return void
	 */
	public static function update() {

		require_once( 'class-ss-wc-mailchimp-pro-migrator.php' );

		SS_WC_MailChimp_Pro_Migrator::migrate( self::version() );

		//Placeholder
		//update_option( 'ss_wc_mailchimp_pro_version', self::version() );

	} //end function update

	/**
	 * Plugin activate function.
	 *
	 * @access public
	 * @static
	 * @param mixed $network_wide
	 * @return void
	 */
	public static function activate( $network_wide = false ) {

		self::update();

	} //end function activate

	/**
	 * Plugin deactivate function.
	 *
	 * @access public
	 * @static
	 * @param mixed $network_wide
	 * @return void
	 */
	public static function deactivate( $network_wide ) {

		// Placeholder

	} //end function deactivate

	/**
	 * Check whether WooCommerce MailChimp Pro is network activated
	 * @since 1.0
	 * @return bool
	 */
	public static function is_network_activated() {
		return is_multisite() && ( function_exists( 'is_plugin_active_for_network' ) && is_plugin_active_for_network( 'woocommerce-mailchimp-pro/woocommerce-mailchimp-pro.php' ) );
	}

	/**
	 * Returns namespace prefixed value
	 * @param  string  $suffix  The suffix to prefix
	 * @return string
	 */
	private function namespace_prefixed( $suffix ) {

		return $this->namespace . '_' . $suffix;

	} // end function namespace_prefixed

} //end final class SS_WC_MailChimp_Plugin
