<?php
/**
 * WooCommerce MailChimp Pro Settings
 *
 * @author 		Saint Systems
 * @package     WooCommerce MailChimp Pro
 * @version		2.0
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'SS_WC_MailChimp_Pro_Settings' ) ) {

	/**
	 * @class   SS_WC_MailChimp_Pro_Settings
	 */
	final class SS_WC_MailChimp_Pro_Settings {

		/**
		 * Singleton instance
		 * @var SS_WC_MailChimp_Pro_Settings
		 */
		private static $instance;

		/**
		 * Main plugin singleton instance
		 * @var SS_WC_MailChimp_Plugin
		 */
		protected $sswcmc;

		/**
		 * Pro plugin singleton instance
		 * @var SS_WC_MailChimp_Pro_Plugin
		 */
		protected $sswcmcpro;

		/**
		 * Singleton instance
		 *
		 * @return SS_WC_MailChimp_Pro_Settings   SS_WC_MailChimp_Pro_Settings object
		 */
		public static function get_instance() {

			if ( empty( self::$instance ) ) {
				self::$instance = new self;
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * @access public
		 * @return void
		 */
		public function __construct() {

			$this->id         = 'mailchimp';
			$this->namespace  = 'ss_wc_' . $this->id;
			$this->label      = __( 'MailChimp', 'woocommerce-mailchimp' );
			$this->sswcmc     = SSWCMC();
			$this->sswcmcpro  = SSWCMCPRO();

			$this->register_hooks();

			// Add hooks and trigger a reload of the settings to pull in merge tags settings
			$this->sswcmc->settings( true );

		} //end function __construct

		/**
		 * Register plugin hooks
		 *
		 * @access public
		 * @return void
		 */
		public function register_hooks() {

			if ( is_admin() ) {

				// Add merge tags settings
				add_filter( 'ss_wc_mailchimp_default_settings', array( $this, 'add_pro_settings_to_default_settings' ), 10 , 1 );

				// Hook in to add the MailChimp tab
				add_filter( 'woocommerce_get_sections_mailchimp', array( $this, 'get_sections' ), 20, 1 );
				add_filter( 'woocommerce_get_settings_mailchimp', array( $this, 'add_pro_section_settings' ), 20, 2 );

				add_action( 'woocommerce_admin_field_ss_wc_mailchimp_pro_license_key', array( $this, 'render_license_key_field' ), 10, 1 );

				add_action( 'woocommerce_admin_field_ss_wc_mailchimp_pro_hidden', array( $this, 'render_hidden_field' ), 10, 1 );
				add_filter( 'ss_wc_mailchimp_settings_general_after_interest_groups', array( $this, 'add_merge_tags' ), 20, 2 );
				add_filter( 'ss_wc_mailchimp_settings_general_after_display_opt_in', array( $this, 'add_subscribe_to_product_lists' ), 20, 2 );

				add_action( 'woocommerce_admin_field_ss_wc_mailchimp_merge_tags', array( $this, 'render_merge_tags_field' ), 10, 1 );

				add_action( 'ss_wc_mailchimp_after_settings_enqueue_js', array( $this, 'wc_enqueue_js' ) );

				add_filter( 'woocommerce_product_data_tabs', array( $this, 'add_mailchimp_product_data_tab' ) );
				add_action( 'woocommerce_product_data_panels', array( $this, 'add_mailchimp_product_data_fields' ) );
				add_action( 'woocommerce_process_product_meta', array( $this, 'add_mailchimp_product_data_fields_save' ) );

			}

		} //end function ensure_tab

		/**
		 * Adds pro settings to the default plugin settings
		 * @param array merge of $default_settings and $pro_settings
		 */
		public function add_pro_settings_to_default_settings( $default_settings ) {

			$pro_settings = array(
				'merge_tags' => '',
				//'pro_license_key' => '',
			);

			return array_merge( $default_settings, $pro_settings );

		} //end function add_pro_settings_to_default_settings

		/**
		 * Get sections
		 *
		 * @return array
		 */
		public function get_sections( $sections ) {

			$troubleshooting_section = $sections['troubleshooting'];

			unset( $sections['troubleshooting'] );

			$sections['pro'] = __( 'Pro', 'woocommerce-mailchimp' );
			$sections['troubleshooting'] = $troubleshooting_section;

			return $sections;
		}

		/**
		 * Get settings array
		 *
		 * @return array
		 */
		public function add_merge_tags( $settings ) {

			if ( $this->sswcmc->has_api_key() && $this->sswcmc->has_list() ) {
				$merge_tags = $this->get_merge_tags();
			} else {
				$merge_tags = array();
			}

			// All WooCommerce available fields
			$checkout_fields = array(
				''                            => __('Select a field...', 'woocommerce-mailchimp'),
				'order_billing_first_name'    => __('Billing First Name', 'woocommerce-mailchimp'),
				'order_billing_last_name'     => __('Billing Last Name', 'woocommerce-mailchimp'),
				'order_billing_company'       => __('Billing Company', 'woocommerce-mailchimp'),
				'order_billing_address_1'     => __('Billing Address 1', 'woocommerce-mailchimp'),
				'order_billing_address_2'     => __('Billing Address 2', 'woocommerce-mailchimp'),
				'order_billing_city'          => __('Billing City', 'woocommerce-mailchimp'),
				'order_billing_state'         => __('Billing State', 'woocommerce-mailchimp'),
				'order_billing_postcode'      => __('Billing Postcode', 'woocommerce-mailchimp'),
				'order_billing_country'       => __('Billing Country', 'woocommerce-mailchimp'),
				'order_billing_phone'         => __('Billing Phone', 'woocommerce-mailchimp'),
				'order_shipping_first_name'   => __('Shipping First Name', 'woocommerce-mailchimp'),
				'order_shipping_last_name'    => __('Shipping Last Name', 'woocommerce-mailchimp'),
				'order_shipping_address_1'    => __('Shipping Address 1', 'woocommerce-mailchimp'),
				'order_shipping_address_2'    => __('Shipping Address 2', 'woocommerce-mailchimp'),
				'order_shipping_city'         => __('Shipping City', 'woocommerce-mailchimp'),
				'order_shipping_state'        => __('Shipping State', 'woocommerce-mailchimp'),
				'order_shipping_postcode'     => __('Shipping Postcode', 'woocommerce-mailchimp'),
				'order_shipping_country'      => __('Shipping Country', 'woocommerce-mailchimp'),
				'order_shipping_method_title' => __('Shipping Method Title', 'woocommerce-mailchimp'),
				'order_payment_method_title'  => __('Payment Method Title ', 'woocommerce-mailchimp'),
				'order_user_id'               => __('User ID', 'woocommerce-mailchimp'),
				'user_first_name'             => __('User First Name', 'woocommerce-mailchimp'),
				'user_last_name'              => __('User Last Name', 'woocommerce-mailchimp'),
				'user_nickname'               => __('User Nickname', 'woocommerce-mailchimp'),
				'user_paying_customer'        => __('User Is Paying Customer', 'woocommerce-mailchimp'),
				'user__order_count'           => __('User Completed Order Count', 'woocommerce-mailchimp'),
			);

			// Add pro feature into general settings for merge tag mapping
			// $settings[] = array(
			// 		'id'          => 'ss_wc_mailchimp_merge_tags',
			// 		'title'       => __( 'Merge Tags', 'woocommerce-mailchimp' ),
			// 		'type'        => 'ss_wc_mailchimp_merge_tags',
			// 		'desc'        => __( 'Map the MailChimp Merge Tags on the left to the WooCommerce Fields on the right. EMAIL is always sent.', 'woocommerce-mailchimp' ),
			// 		'default'     => '',
			// 		'options'     => array(
			// 			'left' => $merge_tags,
			// 			'right' => $checkout_fields
			// 		),
			// 		'class'       => 'wc-enhanced-select',
			// 		// 'custom_attributes' => array(
			// 		// 	'placeholder' => __( 'Select merge tags...', 'woocommerce-mailchimp' ),
			// 		// ),
			// 		'css'         => 'min-width: 200px;',
			// 		'desc_tip'    =>  true,
			// 	);

			return $settings;

		} //end function add_merge_tags

		/**
		 * Get settings array
		 *
		 * @return array
		 */
		public function add_subscribe_to_product_lists( $settings ) {

			$settings[] = array(
				'id'          => 'ss_wc_mailchimp_pro_always_subscribe_to_product_lists',
				'title'       => __( 'Subscribe to Product Lists', 'woocommerce-mailchimp' ),
				'desc'        => __( 'Always', 'woocommerce-mailchimp' ),
				'type'        => 'checkbox',
				'default'     => 'no',
				'desc_tip'    => __( 'If enabled, customers will always be subscribed to product-specific lists, groups, and tags regardless of the "Subscribe Customers" setting above. NOTE: This should ONLY be used for transactional email, not marketing email where explicit permission is required.', 'woocommerce-mailchimp' ),
			);

			return $settings;

		} //end function add_subscribe_to_product_lists

		/**
		 * Add pro section settings array
		 *
		 * @return array
		 */
		public function add_pro_section_settings( $settings, $current_section = '' ) {

			if ( 'pro' === $current_section ) {

				// Add pro feature into general settings for merge tag mapping
				$settings[] = array(
						'title'       => __( 'Pro', 'woocommerce-mailchimp' ),
						'type'        => 'title',
						'desc'        => __( 'Enter your WooCommerce MailChimp Pro license key below to enable automatic updates.', 'woocommerce-mailchimp' ),
						'id'          => 'pro_options',
					);
				$settings[] = array(
						'id'          => 'ss_wc_mailchimp_pro_license_key',
						'title'       => __( 'License Key', 'woocommerce-mailchimp' ),
						'type'        => 'ss_wc_mailchimp_pro_license_key',
						'desc'        => __( 'Your WooCommerce MailChimp Pro License key', 'woocommerce-mailchimp' ),
						'default'     => '',
						'css'         => 'min-width: 350px',
						'desc_tip'    =>  true,
					);
				$settings[] = array(
						'id'                  => 'ss_wc_mailchimp_pro_license_key_status',
						'type'                => 'ss_wc_mailchimp_pro_hidden',
						'default'             => '',
					);
				$settings[] = array(
						'id'                  => 'ss_wc_mailchimp_pro_license_key_response',
						'type'                => 'ss_wc_mailchimp_pro_hidden',
						'default'             => '',
					);
				$settings[] = array(
						'type'        => 'sectionend',
						'id'          => 'pro_options'
					);

			}

			return $settings;

		} //end function add_pro_section_settings

		/**
		 * get_merge_tags function.
		 *
		 * @access public
		 * @return void
		 */
		public function get_merge_tags() {

			$merge_tags = false;

			if ( $this->sswcmc->mailchimp() && $this->sswcmc->has_list() ) {
				$merge_tags = $this->sswcmc->mailchimp()->get_merge_fields( $this->sswcmc->get_list() );
			}

			return $merge_tags;

		} //end function get_merge_tags

		/**
		 * [render_merge_tags_field description]
		 * @return [type] [description]
		 */
		public function render_merge_tags_field( $value ) {

			$custom_attributes = array();

			$required_merge_tags = array_filter( $value['options']['left'], function( $merge_tag ) {
				return $merge_tag['required'];
			});

			$optional_merge_tags = array_filter( $value['options']['left'], function( $merge_tag ) {
				return !$merge_tag['required'];
			});

			$optional_default = array(
					'' => array(
						'name' => __( 'Select a tag...', 'woocommerce-mailchimp' ),
					),
				);
			$optional_merge_tags = array_merge( $optional_default, $optional_merge_tags );

			$optional_merge_tags = array_merge( $optional_default, $optional_merge_tags );

			$option_id = $this->unprefixed( $value['id'] );

			$option_value = $this->sswcmc->settings()[$option_id];

			$selected_required_woocommerce_fields = isset( $option_value['required_woocommerce_fields'] ) ? $option_value['required_woocommerce_fields'] : array();

			$selected_optional_woocommerce_fields = isset( $option_value['optional_woocommerce_fields'] ) ? $option_value['optional_woocommerce_fields'] : array();

			$selected_optional_merge_tags = isset( $option_value['optional_merge_tags'] ) ? $option_value['optional_merge_tags'] : array();

			// Description handling
			$field_description = WC_Admin_Settings::get_field_description( $value );
			extract( $field_description );
			?>
			<tr valign="top">
				<th scope="row" class="titledesc">
					<label for="<?php echo esc_attr( $value['id'] ); ?>"><?php echo esc_html( $value['title'] ); ?></label>
					<?php echo $tooltip_html; ?>
				</th>
				<td class="forminp forminp-<?php echo sanitize_title( $value['type'] ) ?>">
					<table id="ss_wc_mailchimp_merge_tags_table" class="woocommerce-mailchimp-merge-tags-table">
						<thead>
							<tr>
								<th class="sswcmc_mailchimp_merge_tag_column"><?php _e( 'MailChimp Merge Tag', 'woocommerce-mailchimp' ) ?></th>
								<th class="sswcmc_woocommerce_field_column"><?php _e( 'WooCommerce Field', 'woocommerce-mailchimp' ) ?></th>
							</tr>
						</thead>
						<tbody>
						<?php $i = 0; foreach( $required_merge_tags as $req_key => $req_val ):
						?>
							<tr id="ss_wc_mailchimp_required_merge_tags_<?php echo $i ?>"class="required">
								<td>
									<input type="hidden" name="<?php echo esc_attr( $value['id'] ); ?>[required_merge_tags][]" value="<?php echo esc_attr( $req_key ); ?>" />
									<?php echo $req_val['name'] ?> <span class="required">*</span>
								</td>
								<td>
									<select
										name="<?php echo esc_attr( $value['id'] ); ?>[required_woocommerce_fields][]"
										id="<?php echo esc_attr( $value['id'] ); ?>"
										style="<?php echo esc_attr( $value['css'] ); ?>"
										class="<?php echo esc_attr( $value['class'] ); ?>"
										<?php echo implode( ' ', $custom_attributes ); ?>
										<?php
											foreach ( $value['options']['right'] as $key => $val ) {
												?>
												<option value="<?php echo esc_attr( $key ); ?>" <?php

													if ( isset( $selected_required_woocommerce_fields[$i] ) ) {
														selected( $selected_required_woocommerce_fields[$i], $key );
													} //else {
													//	selected( $option_value, $key );
													//}

												?>><?php echo $val ?></option>
												<?php
											}
										?>
									</select>
								</td>
							</tr>
						<?php $i++; endforeach; ?>
						<?php for ( $i = 0; $i < max( 1, count( $selected_optional_merge_tags ) ); $i++ ):
						?>
							<tr id="ss_wc_mailchimp_optional_merge_tags_<?php echo $i ?>" class="optional">
								<td>
									<select
										name="<?php echo esc_attr( $value['id'] ); ?>[optional_merge_tags][]"
										id="<?php echo esc_attr( $value['id'] ); ?>"
										style="<?php echo esc_attr( $value['css'] ); ?>"
										class="<?php echo esc_attr( $value['class'] ); ?>"
										<?php echo implode( ' ', $custom_attributes ); ?>
										<?php
											foreach ( $optional_merge_tags as $key => $val ) {
												?>
												<option value="<?php echo esc_attr( $key ); ?>" <?php

													if ( isset( $selected_optional_merge_tags[$i] ) ) {
														selected( $selected_optional_merge_tags[$i], $key );
													} //else {
													//	selected( $option_value, $key );
													//}

												?>><?php echo $val['name'] ?></option>
												<?php
											}
										?>
									</select>
								</td>
								<td>
									<select
										name="<?php echo esc_attr( $value['id'] ); ?>[optional_woocommerce_fields][]"
										id="<?php echo esc_attr( $value['id'] ); ?>"
										style="<?php echo esc_attr( $value['css'] ); ?>"
										class="<?php echo esc_attr( $value['class'] ); ?>"
										<?php echo implode( ' ', $custom_attributes ); ?>
										<?php
											foreach ( $value['options']['right'] as $key => $val ) {
												?>
												<option value="<?php echo esc_attr( $key ); ?>" <?php

													if ( isset( $selected_optional_woocommerce_fields[$i] ) ) {
														selected( $selected_optional_woocommerce_fields[$i], $key );
													} //else {
													//	selected( $option_value, $key );
													//}

												?>><?php echo $val ?></option>
												<?php
											}
										?>
									</select>
									<a class="ss_wc_mailchimp_delete_merge_tag">
										<i class="dashicons dashicons-no"></i>
									</a>
								</td>
							</tr>
						<?php endfor; ?>
						</tbody>
						<tfoot>
							<tr>
								<td>
									<button type="button" name="ss_wc_mailchimp_add_merge_tag" id="ss_wc_mailchimp_add_merge_tag" class="button"><i class="fa fa-plus"></i>&nbsp;&nbsp;Add Merge Tag
									</button>
								</td>
								<td></td>
							</tr>
						</tfoot>
					</table>
				</td>
			</tr>
			<?php

		} //end function render_merge_tags_field

		/**
		 * [render_license_key_field description]
		 * @return [type] [description]
		 */
		public function render_license_key_field( $value ) {

			$settings = $this->sswcmcpro->settings( true );
			$license_key = $settings['license_key'];
			$license_key_status = $settings['license_key_status'];
			$license_key_response = $settings['license_key_response'];

			if ( !empty( $license_key ) ) {

				$response = $this->sswcmcpro->get_license_handler()->license_call( array(
					'license'    => $license_key,
					'edd_action' => 'check_license',
					'field_id'   => 'edd-check',
					'update'     => true,
				) );

				$settings           = $this->sswcmcpro->settings( true );
				$license_key        = $settings['license_key'];
				$license_key_status = $settings['license_key_status'];
			}

			$type      = $value['type'];
			$option_id = str_replace( 'ss_wc_mailchimp_pro_', '', $value['id'] );

			$option_value = $settings[$option_id];

			if ( 'color' === $value['type'] ) {
				$type = 'text';
				$value['class'] .= 'colorpick';
				$description .= '<div id="colorPickerDiv_' . esc_attr( $value['id'] ) . '" class="colorpickdiv" style="z-index: 100;background:#eee;border:1px solid #ccc;position:absolute;display:none;"></div>';
			}
			// Custom attribute handling
			$custom_attributes = array();

			if ( ! empty( $value['custom_attributes'] ) && is_array( $value['custom_attributes'] ) ) {
				foreach ( $value['custom_attributes'] as $attribute => $attribute_value ) {
					$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
				}
			}

			// Description handling
			$field_description = WC_Admin_Settings::get_field_description( $value );
			extract( $field_description );

			?>
			<tr valign="top">
				<th scope="row" class="titledesc">
					<label for="<?php echo esc_attr( $value['id'] ); ?>"><?php echo esc_html( $value['title'] ); ?></label>
					<?php echo $tooltip_html; ?>
				</th>
				<td class="forminp forminp-<?php echo sanitize_title( $value['type'] ) ?>">
					<?php
					if ( 'color' == $value['type'] ) {
						echo '<span class="colorpickpreview" style="background: ' . esc_attr( $option_value ) . ';"></span>';
					}
					?>
					<input
						name="<?php echo esc_attr( $value['id'] ); ?>"
						id="<?php echo esc_attr( $value['id'] ); ?>"
						type="<?php echo esc_attr( $type ); ?>"
						style="<?php echo esc_attr( $value['css'] ); ?>"
						value="<?php echo esc_attr( $option_value ); ?>"
						class="<?php echo esc_attr( $value['class'] ); ?>"
						placeholder="<?php echo esc_attr( $value['placeholder'] ); ?>"
						<?php echo implode( ' ', $custom_attributes ); ?>
						/>
						<?php if ( ! empty( $option_value ) ) : ?>
						<div class="sswcmcpro-button-wrapper">
							<input type="button" name="_sswcmcpro_setting_edd-activate" value="Activate License" class="button-primary button sswcmcpro-edd-action hide sswcmcpro-setting sswcmcpro-button" data-pending_text="Verifying license…" data-edd_action="activate_license" style="margin-left: 10px;" id="edd-activate">
							<input type="button" name="_sswcmcpro_setting_edd-deactivate" value="Deactivate License" class="button-primary button sswcmcpro-edd-action hide sswcmcpro-setting sswcmcpro-button" data-pending_text="Deactivating license…" data-edd_action="deactivate_license" style="margin-left: 10px;" id="edd-deactivate">
							<input type="button" name="_sswcmcpro_setting_edd-check" value="Check License" class="button-secondary button sswcmcpro-edd-action hide sswcmcpro-setting sswcmcpro-button" data-pending_text="Verifying license…" title="Check the license before saving it" data-edd_action="check_license" style="margin-left: 10px;" id="edd-check">
						</div>
						<?php endif; ?>
						<?php echo $description; ?>
				</td>
			</tr>
			<?php
			// $license_key_field = '<input type="text" name="_sswcmcpro_setting_license_key" id="license_key" class="regular-text sswcmcpro-setting sswcmcpro-text edd-license-key" value="' . $license_key . '" />';

			// $license_key_field .= $this->sswcmcpro->License_Handler->settings_edd_license_activation( false );

			// echo apply_filters( 'ss_wc_mailchimp_pro_license_key_field', $license_key_field );

		} //end function render_license_key_field

		/**
		 * [render_hidden_field description]
		 * @return [type] [description]
		 */
		public function render_hidden_field( $value ) {

			$settings = $this->sswcmcpro->settings( true );

			$type         = $value['type'];
			$option_id = str_replace( 'ss_wc_mailchimp_pro_', '', $value['id'] );

			$option_value = $settings[$option_id];

			?>
			<input
				name="<?php echo esc_attr( $value['id'] ); ?>"
				id="<?php echo esc_attr( $value['id'] ); ?>"
				type="hidden"
				value="<?php echo esc_attr( ( is_array( $option_value ) ? json_encode( $option_value ) : $option_value ) ); ?>"
				/>
			<?php

		} //end function render_hidden_field

		public function get_license_settings() {

			return get_option( 'ss_wc_mailchimp_pro_license', true );

		}

		/**
		 * Save MailChimp data with WooCommerce Product
		 * @param [type] $post_id [description]
		 */
		function add_mailchimp_product_data_fields_save( $post_id ) {

			if ( ! isset( $_POST['ss_wc_mailchimp_list'] ) ) return;
			$mailchimp_lists = $_POST['ss_wc_mailchimp_list'];
			update_post_meta( $post_id, '_ss_wc_mailchimp_list', $mailchimp_lists );

			$synced_lists = $_POST['ss_wc_mailchimp_subscription_sync'];
			update_post_meta( $post_id, '_ss_wc_mailchimp_subscription_sync', $synced_lists );

			$expiration_actions = array();

			foreach ( $mailchimp_lists as $list_id ) {

				$list_expiration_action_key = "ss_wc_mailchimp_list_{$list_id}_expiration_action";

				if ( isset( $_POST[ $list_expiration_action_key ] ) ) {

					$expiration_actions[ $list_id ] = $_POST["ss_wc_mailchimp_list_{$list_id}_expiration_action"];

				}

			}

			update_post_meta( $post_id, '_ss_wc_mailchimp_list_expiration_actions', $expiration_actions );

			// For each selected mailchimp list, check for selected interest groups
			$interest_groups = array();

			foreach ( $mailchimp_lists as $list_id ) {

				$list_interest_groups_key = "ss_wc_mailchimp_list_{$list_id}_interest_groups";

				if ( isset( $_POST[ $list_interest_groups_key ] ) ) {

					$interest_groups[ $list_id ] = $_POST["ss_wc_mailchimp_list_{$list_id}_interest_groups"];

				}

			}

			// Update the selected interest groups for any selected lists
			update_post_meta( $post_id, '_ss_wc_mailchimp_list_interest_groups', $interest_groups );

			$tags = array();

			foreach ( $mailchimp_lists as $list_id ) {

				$list_tags_key = "ss_wc_mailchimp_list_{$list_id}_tags";

				if ( isset( $_POST[ $list_tags_key ] ) ) {

					$tags[ $list_id ] = $_POST["ss_wc_mailchimp_list_{$list_id}_tags"];

				}

			}

			// Update the selected interest groups for any selected lists
			update_post_meta( $post_id, '_ss_wc_mailchimp_list_tags', $tags );

			$double_opt_in_lists = $_POST['ss_wc_mailchimp_list_double_opt_in'];
			update_post_meta( $post_id, '_ss_wc_mailchimp_list_double_opt_in', $double_opt_in_lists );

		} //end function add_mailchimp_product_data_fields_save

		/**
		 * Add the MailChimp tab to WooCommerce products
		 * @param [mixed] $product_data_tabs [existing product data tabs array]
		 * @return [mixed] $product_data_tabs
		 */
		public function add_mailchimp_product_data_tab( $product_data_tabs ) {

			$product_data_tabs['mailchimp'] = array(
				'label'  => __( 'Mailchimp', 'ss_wc_mailchimp' ),
				'target' => 'mailchimp_product_data',
				'class'  => array( 'hide_if_grouped' ),
			);

			return $product_data_tabs;

		} //end function add_mailchimp_product_data_tab

		/**
		 * Adds the mailchimp fields to the WooCommerce product
		 * @return void
		 */
		public function add_mailchimp_product_data_fields() {

			global $woocommerce, $post;

			?>
			<div id="mailchimp_product_data" class="panel woocommerce_options_panel">

				<?php

				echo '<div class="options_group">';
					echo '<p>';
						_e( 'Select the lists and optional interest groups and tags to assign to purchasers of this product.', 'woocommerce-mailchimp' );
					echo '</p>';
				echo '</div>';

				$lists = $this->sswcmc->mailchimp()->get_lists();
				$list_web_ids = $this->sswcmc->mailchimp()->get_list_web_ids();

				$saved_lists = get_post_meta( $post->ID, '_ss_wc_mailchimp_list', true ) ?: array();

				$double_opt_in_lists = get_post_meta( $post->ID, '_ss_wc_mailchimp_list_double_opt_in', true ) ?: array();

				$synced_lists = get_post_meta( $post->ID, '_ss_wc_mailchimp_subscription_sync', true ) ?: array();

				$is_subscription_product = false;

				$product_factory = new WC_Product_Factory();
				$product = $product_factory->get_product( $post->ID );
				if ( is_a( $product, 'WC_Product_Subscription' ) || is_a( $product, 'WC_Product_Variable_Subscription' ) ) {
					$is_subscription_product = true;
				}

				$saved_interest_groups = get_post_meta( $post->ID, '_ss_wc_mailchimp_list_interest_groups', true ) ?: array();

				$saved_tags = get_post_meta( $post->ID, '_ss_wc_mailchimp_list_tags', true ) ?: array();

				$saved_expiration_actions = get_post_meta( $post->ID, '_ss_wc_mailchimp_list_expiration_actions', true ) ?: array();

				foreach( $lists as $list_id => $list_name ) {

					echo '<div class="options_group">';

					woocommerce_wp_checkbox( array(
						'id'            => 'ss_wc_mailchimp_list',
						'name'          => 'ss_wc_mailchimp_list[]',
						// 'wrapper_class' => 'show_if_simple',
						'label'         => __( 'Mailchimp List', 'woocommerce-mailchimp' ),
						'cbvalue'		=> $list_id,
						'value'			=> in_array( $list_id, $saved_lists ) ? $list_id : null,
						'description'   => $list_name,
						'default'  		=> '0',
						'desc_tip'    	=> false,
					) );

					$interest_groups = $this->sswcmc->mailchimp()->get_interest_categories_with_interests( $list_id );

					$this->woocommerce_wp_select( array(
						'id'          => "ss_wc_mailchimp_list_{$list_id}_interest_groups",
						'name'        => "ss_wc_mailchimp_list_{$list_id}_interest_groups[]",
						'label'       => __( 'Interest Groups', 'woocommerce-mailchimp' ),
						'options'     => $interest_groups,
						'style'       => 'min-width:300px;',
						'class'       => 'wc-enhanced-select',
						'value'       => array_key_exists( $list_id, $saved_interest_groups ) ? $saved_interest_groups[ $list_id ] : null,
						'description' => sprintf( __( 'Optional interest groups to assign to purchasers of this product. %sManage groups%s for this list.', 'woocommerce-mailchimp' ), '<a href="https://admin.mailchimp.com/lists/dashboard/groups?id='.$list_web_ids[$list_id].'" target="_blank">', '</a>' ),
						'custom_attributes' => array( 'multiple' => 'multiple' ),
						'desc_tip'    => false,
					) );

					$tags = $this->sswcmc->mailchimp()->get_tags( $list_id );

					$this->woocommerce_wp_select( array(
						'id'          => "ss_wc_mailchimp_list_{$list_id}_tags",
						'name'        => "ss_wc_mailchimp_list_{$list_id}_tags[]",
						'label'       => __( 'Tags', 'woocommerce-mailchimp' ),
						'options'     => $tags,
						'style'       => 'min-width:300px;',
						'class'       => 'wc-enhanced-select',
						'value'       => array_key_exists( $list_id, $saved_tags ) ? $saved_tags[ $list_id ] : null,
						'description' => sprintf( __( 'Optional tags to assign to purchasers of this product. %sManage tags%s for this list.', 'woocommerce-mailchimp' ), '<a href="https://admin.mailchimp.com/lists/tags?id='.$list_web_ids[$list_id].'" target="_blank">', '</a>' ),
						'custom_attributes' => array( 'multiple' => 'multiple' ),
						'desc_tip'    => false,
					) );

					woocommerce_wp_checkbox( array(
						'id'            => "ss_wc_mailchimp_list_{$list_id}_double_opt_in",
						'name'          => "ss_wc_mailchimp_list_double_opt_in[]",
						// 'wrapper_class' => 'show_if_simple',
						'label'         => __( 'Require Double Opt-In', 'woocommerce-mailchimp' ),
						'cbvalue'		=> $list_id,
						'value'			=> in_array( $list_id, $double_opt_in_lists ) ? $list_id : null,
						'description'   => __('Check this box if purchasers of this product should be required to confirm their subscription to this list.', 'woocommerce-mailchimp'),
						'default'  		=> '0',
						'desc_tip'    	=> true,
					) );

					if ( $is_subscription_product ) {
						woocommerce_wp_checkbox( array(
							'id'            => "ss_wc_mailchimp_{$list_id}_subscription_sync",
							'name'          => 'ss_wc_mailchimp_subscription_sync[]',
							'wrapper_class' => 'require_subscription',
							'label'         => __( 'Require subscription?', 'woocommerce-mailchimp' ),
							'cbvalue'		=> $list_id,
							'value'			=> in_array( $list_id, $synced_lists ) ? $list_id : null,
							'description'   => __(
							'Check this box if purchasers are required to have an active product subscription in order to maintain membership to this Mailchimp list, groups, and tags.', 'woocommerce-mailchimp' ),
							'default'       => '1',
							'desc_tip'    	=> true,
						) );

						woocommerce_wp_select( array(
							'id'            => "ss_wc_mailchimp_list_{$list_id}_expiration_action",
							'name'          => "ss_wc_mailchimp_list_{$list_id}_expiration_action",
							'wrapper_class' => 'expiration_action',
							'label'         => __( 'Expiration Action', 'woocommerce-mailchimp' ),
							'value'         => array_key_exists( $list_id, $saved_expiration_actions ) ? $saved_expiration_actions[ $list_id ] : null,
							'options'       => array(
								'0' => 'Unsubscribe from Mailchimp list.',
								'1' => 'Remove selected Interest Groups and Tags.',
								'2' => 'Remove selected Interest Groups only.',
								'3' => 'Remove selected Tags only.',
							),
							'description'   => __(
							'Control the desired action when the customer no longer has an active subscription. "Require subscription" must be checked above.', 'woocommerce-mailchimp' ),
							'default'       => '0',
							'desc_tip'    	=> true,
						) );
					}

					echo '</div>';

				}

				?>

			</div>
			<?php

		} //end function add_mailchimp_product_data_fields

		/**
		 * Based on woocommerce_wp_select /woocommerce/includes/admin/wc-meta-box-function.php
		 * Updated to handle multi-select
		 *
		 * @param array $field
		 */
		private function woocommerce_wp_select( $field ) {

			global $thepostid, $post;

			$thepostid              = empty( $thepostid ) ? $post->ID : $thepostid;
			$field['class']         = isset( $field['class'] ) ? $field['class'] : 'select short';
			$field['style']         = isset( $field['style'] ) ? $field['style'] : '';
			$field['wrapper_class'] = isset( $field['wrapper_class'] ) ? $field['wrapper_class'] : '';
			$field['value']         = isset( $field['value'] ) ? $field['value'] : get_post_meta( $thepostid, $field['id'], true );
			$field['name']          = isset( $field['name'] ) ? $field['name'] : $field['id'];

			// Custom attribute handling
			$custom_attributes = array();

			if ( ! empty( $field['custom_attributes'] ) && is_array( $field['custom_attributes'] ) ) {

				foreach ( $field['custom_attributes'] as $attribute => $value ){
					$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $value ) . '"';
				}
			}

			echo '<p class="form-field ' . esc_attr( $field['id'] ) . '_field ' . esc_attr( $field['wrapper_class'] ) . '"><label for="' . esc_attr( $field['id'] ) . '">' . wp_kses_post( $field['label'] ) . '</label><select id="' . esc_attr( $field['id'] ) . '" name="' . esc_attr( $field['name'] ) . '" class="' . esc_attr( $field['class'] ) . '" style="' . esc_attr( $field['style'] ) . '" ' . implode( ' ', $custom_attributes ) . '>';

			foreach ( $field['options'] as $key => $value ) {
				echo '<option value="' . esc_attr( $key ) . '" ' . ( is_array( $field['value'] ) ?
												selected( in_array( $key, $field['value'] ), true, false )
											:
												selected( esc_attr( $field['value'] ), esc_attr( $key ), false ) ) . '>' . esc_html( $value ) . '</option>';
			}

			echo '</select> ';

			if ( ! empty( $field['description'] ) ) {

				if ( isset( $field['desc_tip'] ) && false !== $field['desc_tip'] ) {
					echo wc_help_tip( $field['description'] );
				} else {
					echo '<span class="description">' . wp_kses_post( $field['description'] ) . '</span>';
				}
			}
			echo '</p>';

		} //end function woocommerce_wp_select

		public function wc_enqueue_js() {
			wc_enqueue_js("
				(function($){

					$(document).ready(function() {
						SS_WC_MailChimp_Pro.init();
					});

				})(jQuery);
			");
		}

		private function unprefixed( $value ) {
			return str_replace( $this->namespace . '_', '', $value );
		}

		// /**
		//  * Inform the user they don't have any MailChimp lists
		//  */
		// public function mailchimp_no_lists_found() {
		// 	echo $this->get_message( sprintf( __( 'Oops! There are no lists in your MailChimp account. %sClick here%s to create one.', 'woocommerce-mailchimp' ), '<a href="https://admin.mailchimp.com/lists/new-list/" target="_blank">', '</a>' ) );
		// }

		// /**
		//  * Display message to user if there is an issue with the MailChimp API call
		//  *
		//  * @since 1.0
		//  * @param void
		//  * @return html the message for the user
		//  */
		// public function mailchimp_api_error_msg() {
		// 	echo $this->get_message(
		// 		sprintf( __( 'Unable to load lists from MailChimp: (%s) %s. ', 'woocommerce-mailchimp' ), $this->mailchimp()->get_error_code(), $this->mailchimp()->get_error_message() ) .
		// 		sprintf( __( 'Please check your Settings %ssettings%s.', 'woocommerce-mailchimp' ), '<a href="' . WOOCOMMERCE_MAILCHIMP_SETTINGS_URL .'">', '</a>' )
		// 	);
		// } //end function mailchimp_api_error_msg

		/**
		 * Helper log function for debugging
		 *
		 * @since 1.2.2
		 */
		private function log( $message ) {
			if ( $this->sswcmc->debug_enabled() ) {
				$logger = new WC_Logger();

				if ( is_array( $message ) || is_object( $message ) ) {
					$logger->add( 'woocommerce-mailchimp', print_r( $message, true ) );
				}
				else {
					$logger->add( 'woocommerce-mailchimp', $message );
				}
			}
		}

	} //end class SS_WC_MailChimp

} //end if ( ! class_exists( 'SS_WC_MailChimp_Pro_Settings' ) )
