<?php
/**
 * WooCommerce MailChimp Pro Subscriptions
 *
 * @author 		Saint Systems
 * @package     WooCommerce MailChimp Subscriptions
 * @version		1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'SS_WC_MailChimp_Pro_Subscriptions' ) ) {

	/**
	 * @class SS_WC_MailChimp_Pro_Subscriptions
	 */
	final class SS_WC_MailChimp_Pro_Subscriptions {

		/**
		 * Plugin singleton instance
		 * @var SS_WC_MailChimp_Pro_Handler
		 */
		private static $instance = null;

		/**
		 * Constructor
		 *
		 * @access public
		 * @return void
		 */
		public function __construct() {

			$this->id         = 'mailchimp';
			$this->namespace  = 'ss_wc_' . $this->id;
			$this->label      = __( 'MailChimp', 'woocommerce-mailchimp' );
			$this->sswcmc     = SSWCMC();
			$this->register_hooks();

		} //end function __construct

		/**
		 * @return SS_WC_MailChimp_Pro_Subscriptions
		 */
		public static function get_instance() {

			if ( empty( self::$instance ) ) {
				self::$instance = new self;
			}

			return self::$instance;

		}

		/**
		 * Register plugin hooks
		 *
		 * @access public
		 * @return void
		 */
		public function register_hooks() {

			add_action( 'woocommerce_subscription_status_updated', array( $this, 'update_mailchimp_subscription_based_on_woocommerce_subscription_status' ), 10, 3 );
			add_action( 'ss_wc_mailchimp_subscribe_to_subscription_product_list', array( $this, 'subscribe_to_subscription_product_list' ), 10, 2 );

		} //end function ensure_tab

		/**
		 * [update_mailchimp_subscription_based_on_woocommerce_subscription_status description]
		 * @param  [type] $subscription [description]
		 * @param  [type] $new_status   [description]
		 * @param  [type] $old_status   [description]
		 * @return [type]                    [description]
		 */
		public function update_mailchimp_subscription_based_on_woocommerce_subscription_status( $subscription, $new_status, $old_status ) {

			$order = $subscription;

			$order_id = method_exists($order, 'get_id') ? $order->get_id(): $order->id;
			$order_billing_email = method_exists($order, 'get_billing_email') ? $order->get_billing_email(): $order->billing_email;
			$order_billing_first_name = method_exists($order, 'get_billing_first_name') ? $order->get_billing_first_name(): $order->billing_first_name;
			$order_billing_last_name = method_exists($order, 'get_billing_last_name') ? $order->get_billing_last_name(): $order->billing_last_name;

			$merge_tags = array(
				'FNAME' => $order_billing_first_name,
				'LNAME' => $order_billing_last_name
			);

			// Set subscription options
			$subscribe_options = array(
				'email'             => $order_billing_email,
				'merge_tags'      	=> $merge_tags,
				'email_type'        => 'html',
				'double_opt_in'     => $this->sswcmc->double_opt_in(),
			);

			$order_items = $order->get_items();

			foreach( $order_items as $product ) {

				$product_id = $product['product_id'];

				if ( ! $subscription_sync_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_subscription_sync', true ) ) return;

				$double_opt_in_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_list_double_opt_in', true );

				if ( ! is_array( $double_opt_in_lists ) ) {
					$double_opt_in_lists = array();
				}

				// check if the product has a list and optional interest groups
				if ( $product_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_list', true ) ) {

					foreach ( $product_lists as $product_list ) {

						$subscribe_options['double_opt_in'] = false;

						if ( ! in_array( $product_list, $subscription_sync_lists ) ) {
							continue;
						}

						// Change the list
						$subscribe_options['list_id'] = $product_list;

						$expiration_action = '0';

						if ( $product_expiration_actions = get_post_meta( $product_id, '_ss_wc_mailchimp_list_expiration_actions', true ) ) {

							if ( array_key_exists( $product_list, $product_expiration_actions ) ) {

								$expiration_action = $product_expiration_actions[$product_list];

							}

						}

						if ( $product_interest_groups = get_post_meta( $product_id, '_ss_wc_mailchimp_list_interest_groups', true ) ) {

							if ( array_key_exists( $product_list, $product_interest_groups ) ) {

								$interest_group_switch = true;
								if ( 'active' != $new_status && ( $expiration_action == '1' || $expiration_action == '2' ) ) {
									$interest_group_switch = false;
								}

								$product_list_interest_groups = array_fill_keys( $product_interest_groups[$product_list], $interest_group_switch );

								$subscribe_options['interest_groups'] = $product_list_interest_groups;

							}

						}

						if ( $product_tags = get_post_meta( $product_id, '_ss_wc_mailchimp_list_tags', true ) ) {

							if ( array_key_exists( $product_list, $product_tags ) ) {

								$tag_switch = 'active';
								if ( 'active' != $new_status && ( $expiration_action == '1' || $expiration_action == '3' ) ) {
									$tag_switch = 'inactive';
								}

								$product_list_tags = $product_tags[$product_list];

								$tags = $this->sswcmc->mailchimp()->get_tags( $product_list );

								$product_list_tags = array_map( function( $tag ) use ( $tags, $tag_switch ) {
									return array(
										'name' => $tags[$tag],
										'status' => $tag_switch,
									);
								}, $product_list_tags );

								$subscribe_options['tags'] = $product_list_tags;

							}

						}

						if ( in_array( $product_list, $double_opt_in_lists ) ) {
							$subscribe_options['double_opt_in'] = true;
						}

						if ( ! empty( $subscribe_options['list_id'] ) ) {

							$this->log( sprintf( __( __METHOD__ . '(): Queueing subscription to subscription product list for customer (%s) to list %s', 'woocommerce-mailchimp'), $order_billing_email, $product_list ) );

							$as_args = array(
								'subscribe_options' => $subscribe_options,
								'product_id' => $product_id,
								'new_status' => $new_status,
								'old_status' => $old_status,
								'expiration_action' => $expiration_action,
							);
							$hash = md5( json_encode( $as_args ) );
							update_post_meta( $order_id, $hash, $as_args );

							// Queue the subscription to subscription product list.
							as_schedule_single_action( time(), 'ss_wc_mailchimp_subscribe_to_subscription_product_list', array( $order_id, $hash ), 'sswcmc' );

						}

					}

				}

			}

		} //end function maybe_subscribe_to_product_lists

		/**
		 * Subscribe the user to the WooCommerce Subscription Product List.
		 */
		public function subscribe_to_subscription_product_list( $order_id, $hash ) {

			$as_args = get_post_meta( $order_id, $hash, true );
			extract( $as_args );
			delete_post_meta( $order_id, $hash );

			// Allow hooking into subscription options.
			$options = apply_filters( 'ss_wc_mailchimp_pro_subscribe_options', $subscribe_options, $order_id );

			// Extract $options into variables.
			extract( $options );

			if ( $product_tags = get_post_meta( $product_id, '_ss_wc_mailchimp_list_tags', true ) ) {

				if ( array_key_exists( $product_list, $product_tags ) ) {

					$product_list_tags = $product_tags[$product_list];

					$tags = $this->sswcmc->mailchimp()->get_tags( $product_list );

					$product_list_tags = array_map( function( $tag ) use ( $tags ) {
						return array(
							'name' => $tags[$tag],
							'status' => 'active',
						);
					}, $product_list_tags );

					$subscribe_options['tags'] = $product_list_tags;

				}

			}

			// Call API
			if ( 'active' == $new_status ) {
				$this->log( sprintf( __( __METHOD__ . '(): Processing queued subscription to product list for customer (%s) to list %s', 'woocommerce-mailchimp' ), $email, $list_id ) );

				$this->log( sprintf( __( __METHOD__ . '(): Subscribing customer to MailChimp: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );

				$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );
			} else {
				if ( $old_status != 'active' ) return;

				$this->log( sprintf( __( __METHOD__ . '(): Processing queued unsubscribe from product list for customer (%s), list %s', 'woocommerce-mailchimp' ), $email, $list_id ) );

				$order = $this->wc_get_order( $order_id );

				$user_subscriptions = wcs_get_subscriptions(array(
					'subscriptions_per_page' => -1,
					'customer_id' => $order->get_data()['customer_id'],
					'product_id' => $product_id,
					'subscription_status' => 'active',
				));

				$api_response = null;

				if ( count( $user_subscriptions ) == 0 ) {

					switch ( $expiration_action ) {

						// Remove Interest Groups and Tags.
						case '1':
							$this->log( sprintf( __( __METHOD__ . '(): Updating subscriber, removing selected interest groups and tags from subscriber: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );
							$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );
							break;

						// Remove Interest Groups only.
						case '2':
							$this->log( sprintf( __( __METHOD__ . '(): Updating subscriber, removing selected interest groups from subscriber: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );
							$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );
							break;

						// Remove Tags only.
						case '3':
							$this->log( sprintf( __( __METHOD__ . '(): Updating subscriber, removing selected tags from subscriber: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );
							$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );
							break;

						// Unsubscribe from list.
						default:
							$this->log( sprintf( __( __METHOD__ . '(): Unsubscribing customer from MailChimp: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );
							$api_response = $this->sswcmc->mailchimp()->unsubscribe( $list_id, $email );
							break;
					}
				} else {
					$this->log( sprintf( __( __METHOD__ . '(): Customer (%s) still has %s active subscriptions to product ($product_id: %s). Unsubscribe aborted.', 'woocommerce-mailchimp' ), $email, count( $user_subscriptions ), $product_id ) );
				}
			}

			if ( ! empty( $api_response ) ) {
				// Log api response.
				$this->log( sprintf( __( __METHOD__ . '(): MailChimp API response: %s', 'woocommerce-mailchimp' ), print_r( $api_response, true ) ) );
			}

			if ( $api_response === false ) {
				// Format error message
				$error_response = sprintf( __( __METHOD__ . '(): WooCommerce MailChimp subscription failed: %s (%s)', 'woocommerce-mailchimp' ), $this->sswcmc->mailchimp()->get_error_message(), $this->sswcmc->mailchimp()->get_error_code() );

				// Log the error.
				$this->log( $error_response );

				// New hook for failing operations.
				do_action( 'ss_wc_mailchimp_subscription_failed', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );

				// Email admin.
				wp_mail( get_option( 'admin_email' ), __( 'WooCommerce MailChimp subscription failed', 'woocommerce-mailchimp' ), $error_response );
			} else {
				// Hook on success.
				do_action( 'ss_wc_mailchimp_subscription_success', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );
			}
		} //end function subscribe_to_subscription_product_list

		/**
		 * WooCommerce 2.2 support for wc_get_order
		 *
		 * @since 1.0.0
		 *
		 * @access private
		 * @param int $order_id
		 * @return void
		 */
		private function wc_get_order( $order_id ) {
			if ( function_exists( 'wc_get_order' ) ) {
				return wc_get_order( $order_id );
			} else {
				return new WC_Order( $order_id );
			}
		}

		/**
		 * subscribe function.
		 *
		 * @access public
		 * @param int $order_id
		 * @param mixed $first_name
		 * @param mixed $last_name
		 * @param mixed $email
		 * @param string $listid (default: 'false')
		 * @return void
		 */
		public function subscribe( $order_id, $first_name, $last_name, $email, $list_id = 'false' ) {
			if ( ! $email ) {
				return; // Email is required
			}

			if ( 'false' == $list_id ) {
				$list_id = $this->sswcmc->get_list();
			}

			$merge_tags = array(
				'FNAME' => $first_name,
				'LNAME' => $last_name
			);

			$interest_groups = $this->sswcmc->interest_groups();

			if ( ! empty(  $interest_groups) ) {
				$interest_groups = array_fill_keys( $this->sswcmc->interest_groups(), true );

				// Allow hooking into variables
				$interest_groups = apply_filters( 'ss_wc_mailchimp_subscribe_interest_groups', $interest_groups, $order_id, $email );
			}

			// Allow hooking into variables
			$merge_tags = apply_filters( 'ss_wc_mailchimp_subscribe_merge_tags', $merge_tags, $order_id, $email );

			// Set subscription options
			$subscribe_options = array(
				'list_id'           => $list_id,
				'email'             => $email,
				'merge_tags'      	=> $merge_tags,
				'interest_groups'   => $interest_groups,
				'email_type'        => 'html',
				'double_opt_in'     => $this->sswcmc->double_opt_in(),
			);

			// Allow hooking into subscription options
			$options = apply_filters( 'ss_wc_mailchimp_subscribe_options', $subscribe_options, $order_id  );

			// Extract options into variables
			extract( $options );

			// Log
			$this->log( sprintf( __( __METHOD__ . '(): Subscribing customer to MailChimp: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );

			do_action( 'ss_wc_mailchimp_before_subscribe', $subscribe_options, $order_id );

			// Call API
			$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );

			do_action( 'ss_wc_mailchimp_after_subscribe', $subscribe_options, $order_id );

			// Log api response
			$this->log( sprintf( __( __METHOD__ . '(): MailChimp API response: %s', 'woocommerce-mailchimp' ), print_r( $api_response, true ) ) );

			if ( $api_response === false ) {
				// Format error message
				$error_response = sprintf( __( __METHOD__ . '(): WooCommerce MailChimp subscription failed: %s (%s)', 'woocommerce-mailchimp' ), $this->sswcmc->mailchimp()->get_error_message(), $this->sswcmc->mailchimp()->get_error_code() );

				// Log
				$this->log( $error_response );

				// New hook for failing operations
				do_action( 'ss_wc_mailchimp_subscription_failed', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );

				// Email admin
				wp_mail( get_option( 'admin_email' ), __( 'WooCommerce MailChimp subscription failed', 'woocommerce-mailchimp' ), $error_response );
			} else {
				// Hook on success
				do_action( 'ss_wc_mailchimp_subscription_success', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );
			}
		}

		/**
		 * Helper log function for debugging
		 *
		 * @since 1.0.0
		 */
		private function log( $message ) {
			if ( $this->sswcmc->debug_enabled() ) {
				$logger = new WC_Logger();

				if ( is_array( $message ) || is_object( $message ) ) {
					$logger->add( 'woocommerce-mailchimp', print_r( $message, true ) );
				}
				else {
					$logger->add( 'woocommerce-mailchimp', $message );
				}
			}
		}

	} //end class SS_WC_MailChimp_Pro_Handler

} //end if ( ! class_exists( 'SS_WC_MailChimp_Pro_Handler' ) )
