<?php
/**
 * Handle issues with plugin and version compatibility
 *
 * @package   WooCommerce MailChimp Pro
 * @author    Saint Systems, LLC
 * @link      http://www.saintsystems.com
 * @copyright Copyright 2019, Saint Systems, LLC
 *
 * @since 1.0.0
 */

/**
 * Handle WooCommerce MailChimp Pro compatibility notices and fallback shortcodes
 * @since 1.0.0
 */
final class SS_WC_MailChimp_Pro_Compatibility {

	/**
	 * Plugin singleton instance
	 * @var SS_WC_MailChimp_Pro_Compatibility
	 */
	private static $instance = null;

	/**
	 * @var bool Is WooCommerce version valid and is WooCommerce loaded?
	 */
	public static $valid_woocommerce_mailchimp = false;

	/**
	 * @var array Holder for notices to be displayed in frontend shortcodes if not valid
	 */
	static private $notices = array();

	function __construct() {

		self::$valid_woocommerce_mailchimp = self::check_woocommerce_mailchimp();

		$this->add_hooks();
	}

	function add_hooks() {

		add_filter( 'sswcmc.pro.admin.notices', array( $this, 'insert_admin_notices' ) );

	}

	/**
	 * Add the compatibility notices to the other admin notices
	 * @param array $notices
	 *
	 * @return array
	 */
	function insert_admin_notices( $notices = array() ) {

		return array_merge( $notices, self::$notices );

	} //end function insert_admin_notices

	/**
	 * @return SS_WC_MailChimp_Pro_Compatibility
	 */
	public static function get_instance() {

		if ( empty( self::$instance ) ) {
			self::$instance = new self;
		}

		return self::$instance;

	}

	/**
	 * Is everything compatible with this version of the plugin?
	 * @return bool
	 */
	public static function is_valid() {

		return self::is_valid_woocommerce_mailchimp();

	}

	/**
	 * @since 1.0.0
	 * @return bool
	 */
	static function is_valid_woocommerce_mailchimp() {
		return self::$valid_woocommerce_mailchimp;
	}

	/**
	 * Get admin notices
	 * @since 1.0.0
	 * @return array
	 */
	public static function get_notices() {
		return self::$notices;
	}

	/**
	 * Check if WooCommerce MailChimp plugin is active and show notice if not.
	 *
	 * @since 1.0.0
	 *
	 * @access public
	 * @return boolean True: checks have been passed; WooCommerce MailChimp is fine to run; False: checks have failed, don't continue loading
	 */
	public static function check_woocommerce_mailchimp() {

		$woocommerce_mailchimp_is_active = false;

		// Bypass other checks: if the class exists
		if ( defined( 'SS_WC_MAILCHIMP_VERSION' ) && class_exists( 'SS_WC_MailChimp_Plugin' ) ) {

			$woocommerce_mailchimp_is_active = true;
			$woocommerce_mailchimp_version = SS_WC_MAILCHIMP_VERSION;

		} else {

			if ( $wcmc_status = self::get_plugin_status( 'woocommerce-mailchimp/woocommerce-mailchimp.php' ) ) {
				if ( true === $wcmc_status ) {
					$woocommerce_mailchimp_is_active = true;
				}
				$wcmc_data = get_plugin_data( WP_CONTENT_DIR . '/plugins/woocommerce-mailchimp/woocommerce-mailchimp.php' );
				$woocommerce_mailchimp_version = $wcmc_data['Version'];
			}

		}

		if ( true === $woocommerce_mailchimp_is_active ) {
			// and the version's right, we're good.
			if ( true === version_compare( $woocommerce_mailchimp_version, SS_WC_MAILCHIMP_MIN_WCMC_VERSION, ">=" ) ) {
				return true;
			}

			$button = function_exists('is_network_admin') && is_network_admin() ? '<strong><a href="#woocommerce-mailchimp">' : '<strong><a href="'. wp_nonce_url( admin_url( 'update.php?action=upgrade-plugin&plugin=woocommerce-mailchimp/woocommerce-mailchimp.php' ), 'upgrade-plugin_woocommerce-mailchimp/woocommerce-mailchimp.php') . '" class="button button-large">';

			// Or the version's wrong
			self::$notices['wcmc_version'] = array(
				'class' => 'notice-error',
				'message' => sprintf( __( "%sWooCommerce MailChimp Pro requires WooCommerce MailChimp version %s or newer.%s You're using WooCommerce MailChimp version %s. %sUpdate WooCommerce MailChimp%s to use the WooCommerce MailChimp Pro plugin.", 'woocommerce-mailchimp' ), '<h3>', SS_WC_MAILCHIMP_MIN_WCMC_VERSION, "</h3>\n\n", '<span style="font-family: Consolas, Courier, monospace;">'.$woocommerce_mailchimp_version.'</span>', $button, '</strong></a>' )
			);

			return false;
		}

		/**
		 * The plugin is activated and yet somehow WooCommerce MailChimp didn't get picked up...
		 * OR
		 * It's the Network Admin and we just don't know whether the sites have WooCommerce MailChimp activated themselves.
		 */
		if ( $woocommerce_mailchimp_is_active || is_network_admin() ) {
			return true;
		}
		// If WooCommerce MailChimp doesn't exist, assume WooCommerce MailChimp not active
		$return = false;

		switch ( $wcmc_status ) {
			case 'inactive':

				// Required for multisite
				if( ! function_exists('wp_create_nonce') ) {
					require_once ABSPATH . WPINC . '/pluggable.php';
				}

				// Otherwise, throws an error on activation & deactivation "Use of undefined constant LOGGED_IN_COOKIE"
				if( is_multisite() ) {
					wp_cookie_constants();
				}

				$return = false;

				$button = function_exists('is_network_admin') && is_network_admin() ? '<strong><a href="#woocommerce-mailchimp">' : '<strong><a href="'. wp_nonce_url( admin_url( 'plugins.php?action=activate&plugin=woocommerce-mailchimp/woocommerce-mailchimp.php' ), 'activate-plugin_woocommerce-mailchimp/woocommerce-mailchimp.php') . '" class="button button-large">';

				self::$notices['wcmc_inactive'] = array( 'class' => 'error', 'message' => sprintf( __( '%sWooCommerce MailChimp Pro requires WooCommerce MailChimp to be active. %sActivate WooCommerce MailChimp%s to use the WooCommerce MailChimp Pro plugin.', 'woocommerce-mailchimp' ), '<h3>', "</h3>\n\n". $button, '</a></strong>' ) );
				break;
			default:
				$button = function_exists('is_network_admin') && is_network_admin() ? '<strong><a href="#woocommerce-mailchimp">' : '<strong><a href="'. wp_nonce_url( admin_url( 'update.php?action=install-plugin&plugin=woocommerce-mailchimp' ), 'install-plugin_woocommerce-mailchimp' ) . '" class="button button-large">';

				self::$notices['wcmc_installed'] = array( 'class' => 'error', 'message' => sprintf( __( '%sWooCommerce MailChimp Pro requires WooCommerce MailChimp to be installed in order to run properly. %sInstall WooCommerce MailChimp%s to use the WooCommerce MailChimp Pro plugin.', 'woocommerce-mailchimp' ), '<h3>', "</h3>\n\n". $button, '</a></strong>') );
				break;
		}

		return $return;

	} //end public static function check_woocommerce()

	/**
	 * Check if specified plugin is active, inactive or not installed
	 *
	 * @access public
	 * @static
	 * @param string $location (default: '')
	 * @return boolean|string True: plugin is active; False: plugin file doesn't exist at path; 'inactive' it's inactive
	 */
	public static function get_plugin_status( $location = '' ) {

		if ( ! function_exists('is_plugin_active') ) {
			include_once( ABSPATH . '/wp-admin/includes/plugin.php' );
		}

		if ( is_network_admin() && is_plugin_active_for_network( $location ) ) {
			return true;
		}

		if ( !is_network_admin() && is_plugin_active( $location ) ) {
			return true;
		}

		if (
			!file_exists( trailingslashit( WP_PLUGIN_DIR ) . $location ) &&
			!file_exists( trailingslashit( WPMU_PLUGIN_DIR ) . $location )
		) {
			return false;
		}

		return 'inactive';
	}

} //SS_WC_MailChimp_Pro_Compatibility
