<?php
/**
 * WooCommerce MailChimp Pro Handler
 *
 * @author      Saint Systems
 * @package     WooCommerce MailChimp Pro
 * @version     1.0
 */

if ( ! defined( 'ABSPATH' ) ) exit; // Exit if accessed directly

if ( ! class_exists( 'SS_WC_MailChimp_Pro_Handler' ) ) {

	/**
	 * @class SS_WC_MailChimp_Pro_Handler
	 */
	final class SS_WC_MailChimp_Pro_Handler {

		/**
		 * Plugin singleton instance
		 * @var SS_WC_MailChimp_Pro_Handler
		 */
		private static $instance = null;

		/**
		 * Constructor
		 *
		 * @access public
		 * @return void
		 */
		public function __construct() {

			$this->id         = 'mailchimp';
			$this->namespace  = 'ss_wc_' . $this->id;
			$this->label      = __( 'MailChimp', 'woocommerce-mailchimp' );
			$this->sswcmc     = SSWCMC();
			$this->sswcmcpro  = SSWCMCPRO();
			$this->register_hooks();

		} //end function __construct

		/**
		 * @return SS_WC_MailChimp_Handler
		 */
		public static function get_instance() {

			if ( empty( self::$instance ) ) {
				self::$instance = new self;
			}

			return self::$instance;

		}

		/**
		 * Register plugin hooks
		 *
		 * @access public
		 * @return void
		 */
		public function register_hooks() {

			add_action( 'ss_wc_mailchimp_after_subscribe', array( $this, 'queue_maybe_subscribe_to_product_lists' ), 10, 3 );
			add_action( 'ss_wc_mailchimp_maybe_subscribe_to_product_lists', array( $this, 'maybe_subscribe_to_product_lists' ), 10, 2 );
			add_action( 'ss_wc_mailchimp_subscribe_to_product_list', array( $this, 'subscribe_to_product_list' ), 10, 2 );

		} //end function ensure_tab

		/**
		 * [maybe_subscribe_to_product_lists description]
		 * @param  [type] $subscribe_options [description]
		 * @param  [type] $order_id          [description]
		 * @return [type]                    [description]
		 */
		public function queue_maybe_subscribe_to_product_lists( $subscribe_customer, $subscribe_options, $order_id ) {

			$this->log( sprintf( __( __METHOD__ . '(): Queueing subscription ($subscribe_customer: %s) for customer (%s) to list %s for order (%s)', 'woocommerce-mailchimp' ), $subscribe_customer, $subscribe_options['email'], $subscribe_options['list_id'], $order_id ) );

			$subscribe_customer = get_post_meta( $order_id, 'ss_wc_mailchimp_opt_in', true );

			$as_args = array(
				'subscribe_customer' => $subscribe_customer,
				'subscribe_options' => $subscribe_options
			);
			$hash = md5( json_encode( $as_args ) );
			update_post_meta( $order_id, $hash, $as_args );
			$this->log( sprintf( __( __METHOD__ . '(): Order meta updated for $order_id(%s), $hash(%s), $as_args: %s', 'woocommerce-mailchimp' ), $order_id, $hash, print_r( $as_args, true ) ) );

			// Queue the subscription.
			as_schedule_single_action( time(), 'ss_wc_mailchimp_maybe_subscribe_to_product_lists', array( $order_id, $hash ), 'sswcmc' );

		} //end function queue_maybe_subscribe_to_product_lists

		/**
		 * [maybe_subscribe_to_product_lists description]
		 * @param  [type] $subscribe_options [description]
		 * @param  [type] $order_id          [description]
		 * @return [type]                    [description]
		 */
		public function maybe_subscribe_to_product_lists( $order_id, $hash ) {

			$order = $this->wc_get_order( $order_id );

			$as_args = get_post_meta( $order_id, $hash, true );
			$this->log( sprintf( __( __METHOD__ . '(): Order meta retrieved for $order_id(%s), $hash(%s), $as_args: %s', 'woocommerce-mailchimp' ), $order_id, $hash, print_r( $as_args, true ) ) );
			extract( $as_args );
			delete_post_meta( $order_id, $hash );

			$this->log( sprintf( __( __METHOD__ . '(): Processing queue_maybe_subscribe_to_product_lists subscription ($subscribe_customer: %s) for customer (%s) to list %s for order (%s)', 'woocommerce-mailchimp'), $subscribe_customer, $subscribe_options['email'], $subscribe_options['list_id'], $order_id ) );

			$always_subscribe_to_product_lists = $this->sswcmcpro->always_subscribe_to_product_lists();

			$subscribe_customer = !$subscribe_customer || empty( $subscribe_customer ) || 'yes' === $subscribe_customer || $always_subscribe_to_product_lists;

			// Clear out the defaults before we go.
			unset( $subscribe_options['list_id'] );
			unset( $subscribe_options['interest_groups'] );
			$interest_groups = [];

			$order_items = $order->get_items();

			foreach ( $order_items as $item_product ) {

				// Get the product ID.
				$product_id = $item_product->get_product_id();

				// Get the WC_Product object.
				$product = $item_product->get_product();

				// Get the product SKU (using WC_Product method).
				$sku = $product->get_sku();

				// Get the common data in an array.
				$item_product_data_array = $item_product->get_data();

				// Get the special meta data in an array.
				$item_product_meta_data_array = $item_product->get_meta_data();

				$subscription_sync_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_subscription_sync', true );

				$double_opt_in_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_list_double_opt_in', true );

				// $product_id = $product[ 'product_id' ];

				// check if the product has a list and optional interest groups.
				if ( $product_lists = get_post_meta( $product_id, '_ss_wc_mailchimp_list', true ) ) {

					foreach ( $product_lists as $product_list ) {

						$subscribe_options['double_opt_in'] = false;

						if ( is_array( $subscription_sync_lists ) ) {
							// If this is a subscription product, skip it. It will be handled by the SS_WC_MailChimp_Pro_Subscriptions class.
							if ( is_a( $product, 'WC_Product_Subscription' ) && in_array( $product_list, $subscription_sync_lists ) ) {
								continue;
							}
						}

						// Change the list.
						$subscribe_options['list_id'] = $product_list;
						unset( $subscribe_options['interest_groups'] );

						if ( $product_interest_groups = get_post_meta( $product_id, '_ss_wc_mailchimp_list_interest_groups', true ) ) {

							if ( array_key_exists( $product_list, $product_interest_groups ) ) {

								$product_list_interest_groups = array_fill_keys( $product_interest_groups[$product_list], true );

								$subscribe_options['interest_groups'] = $product_list_interest_groups;

							}

						}

						if ( $product_tags = get_post_meta( $product_id, '_ss_wc_mailchimp_list_tags', true ) ) {

							if ( array_key_exists( $product_list, $product_tags ) ) {

								$product_list_tags = $product_tags[$product_list];

								$tags = $this->sswcmc->mailchimp()->get_tags( $product_list );

								$product_list_tags = array_map( function( $tag ) use ( $tags ) {
									return array(
										'name' => $tags[$tag],
										'status' => 'active',
									);
								}, $product_list_tags );

								$subscribe_options['tags'] = $product_list_tags;

							}

						}

						$this->log( sprintf( __METHOD__ . '(): $subscribe_options: (%s)', print_r( $subscribe_options, true ) ) );

						if ( is_array( $double_opt_in_lists ) ) {
							if ( in_array( $product_list, $double_opt_in_lists ) ) {
								$subscribe_options['double_opt_in'] = true;
							}
						}

						if ( ! empty( $subscribe_options['list_id'] ) && $subscribe_customer ) {

							$this->log( sprintf( __( __METHOD__ . '(): Queueing subscribe_to_product_list for customer (%s) to list %s for order (%s)', 'woocommerce-mailchimp' ), $subscribe_options['email'], $subscribe_options['list_id'], $order_id ) );

							$as_args = array(
								'subscribe_options' => $subscribe_options
							);
							$hash = md5( json_encode( $as_args ) );
							update_post_meta( $order_id, $hash, $as_args );
							$this->log( sprintf( __( __METHOD__ . '(): Order meta updated for $order_id(%s), $hash(%s), $as_args: %s', 'woocommerce-mailchimp' ), $order_id, $hash, print_r( $as_args, true ) ) );

							// Queue the subscription.
							as_schedule_single_action( time(), 'ss_wc_mailchimp_subscribe_to_product_list', array( $order_id, $hash ), 'sswcmc' );

						} else {

							$this->log( sprintf( __( __METHOD__ . '(): Cancelled subscribe_to_product_list ($subscribe_customer: %s ) for customer (%s) to list %s for order (%s)', 'woocommerce-mailchimp'), $subscribe_customer, $subscribe_options['email'], $subscribe_options['list_id'], $order_id ) );

						}

					}

				}

			}

		} //end function maybe_subscribe_to_product_lists

		/**
		 * Subscribe to a product list
		 */
		public function subscribe_to_product_list( $order_id, $hash ) {

			$as_args = get_post_meta( $order_id, $hash, true );
			$this->log( sprintf( __( __METHOD__ . '(): Order meta retrieved for $order_id(%s), $hash(%s), $as_args: %s', 'woocommerce-mailchimp' ), $order_id, $hash, print_r( $as_args, true ) ) );
			extract( $as_args );
			delete_post_meta( $order_id, $hash );

			// Allow hooking into subscription options.
			$options = apply_filters( 'ss_wc_mailchimp_pro_subscribe_options', $subscribe_options, $order_id );

			if ( empty( $options ) ) {
				$this->log( sprintf( __( __METHOD__ . '(): $options were empty for $order_id(%s), $hash(%s), $as_args(%s)', 'woocommerce-mailchimp' ), $order_id, $hash, print_r( $as_args, true ) ) );
				return;
			}

			// Extract $options into variables.
			// extract( $options );
			$list_id = !empty ($options['list_id'] ) ? $options['list_id'] : null;
			$email = !empty ($options['email'] ) ? $options['email'] : null;
			$email_type = !empty ($options['email_type'] ) ? $options['email_type'] : null;
			$merge_tags = !empty ($options['merge_tags'] ) ? $options['merge_tags'] : null;
			$interest_groups = !empty ($options['interest_groups'] ) ? $options['interest_groups'] : null;
			$double_opt_in = !empty ($options['double_opt_in'] ) ? $options['double_opt_in'] : null;
			$tags = !empty ($options['tags'] ) ? $options['tags'] : null;

			// Log the call.
			$this->log( sprintf( __( __METHOD__ . '(): Subscribing customer to MailChimp product list: %s', 'woocommerce-mailchimp' ), print_r( $options, true ) ) );

			// Call API.
			$api_response = $this->sswcmc->mailchimp()->subscribe( $list_id, $email, $email_type, $merge_tags, $interest_groups, $double_opt_in, $tags );

			if ( isset( $api_response['_links'] ) ) {
				unset( $api_response['_links'] );
			}

			// Log api response.
			$this->log( sprintf( __( __METHOD__ . '(): MailChimp API response: %s', 'woocommerce-mailchimp'), print_r( $api_response, true ) ) );

			if ( $api_response === false ) {
				// Format error message.
				$error_response = sprintf( __( __METHOD__ . '(): WooCommerce MailChimp subscription failed: %s (%s)', 'woocommerce-mailchimp' ), $this->sswcmc->mailchimp()->get_error_message(), $this->sswcmc->mailchimp()->get_error_code() );

				// Log the error response.
				$this->log( sprintf( __( __METHOD__ . '(): Error returned from MailChimp: %s', 'woocommerce-mailchimp' ), $error_response ) );

				// New hook for failing operations.
				do_action( 'ss_wc_mailchimp_pro_subscription_failed', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );

				// Email admin.
				wp_mail( get_option( 'admin_email' ), __( 'WooCommerce MailChimp subscription failed', 'woocommerce-mailchimp' ), $error_response );
			} else {
				// Hook on success.
				do_action( 'ss_wc_mailchimp_pro_subscription_success', $email, array( 'list_id' => $list_id, 'order_id' => $order_id ) );
			}
		} //end function subscribe_to_product_list

		/**
		 * WooCommerce 2.2 support for wc_get_order
		 *
		 * @since 1.0.0
		 *
		 * @access private
		 * @param int $order_id The order id.
		 */
		private function wc_get_order( $order_id ) {
			if ( function_exists( 'wc_get_order' ) ) {
				return wc_get_order( $order_id );
			} else {
				return new WC_Order( $order_id );
			}
		}

		/**
		 * Helper log function for debugging
		 *
		 * @since 1.0.0
		 */
		private function log( $message ) {
			if ( is_array( $message ) || is_object( $message ) ) {
				do_action( 'sswcmc_log', print_r( $message, true ) );
			} else {
				do_action( 'sswcmc_log', $message );
			}
		}

	} //end class SS_WC_MailChimp_Pro_Handler

} //end if ( ! class_exists( 'SS_WC_MailChimp_Pro_Handler' ) )
